terraform {
  required_providers {
    try-whitesky-cloud = {
      source  = "try-whitesky-cloud/try-whitesky-cloud"
      version = "~> 2.0"
    }
  }
}

provider "try-whitesky-cloud" {
  client_jwt         = var.client_jwt
  client_api_timeout = 300
}

resource "try-whitesky-cloud_cloudspace" "cs" {
  customer_id         = var.customer_id
  location            = var.location
  name                = var.cloudspace_name
  external_network_id = var.external_network_id
  private             = false
}

# Data definition for image
data "try-whitesky-cloud_image" "image" {
  most_recent = true
  name_regex  = "(?i).*ubuntu.*18"
  customer_id = var.customer_id
  location    = var.location
}

# Definition of the vm to be created with the settings defined in terraform.tfvars
resource "try-whitesky-cloud_machine" "kube-mgt" {
  customer_id   = var.customer_id
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  image_id      = data.try-whitesky-cloud_image.image.id
  memory        = var.memory
  vcpus         = var.vcpus
  disk_size     = var.disksize
  name          = "${var.cloudspace_name}-terraform-kube-mgt"
  description   = "${var.vm_description} - management node"
  userdata      = "users: [{name: ansible, shell: /bin/bash, ssh-authorized-keys: [${var.ssh_key_pub}]}, {name: root, shell: /bin/bash, ssh-authorized-keys: [${var.ssh_key_pub}]}]"
}

output "kube-mgt" {
  value = try-whitesky-cloud_cloudspace.cs.external_network_ip
}

# SSH access to mgt vm
resource "try-whitesky-cloud_port_forwarding" "mgt-ssh" {
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  customer_id   = var.customer_id
  vm_id         = try-whitesky-cloud_machine.kube-mgt.id
  public_port   = 2222
  local_port    = 22
  protocol      = "tcp"
}

# Master machines
resource "try-whitesky-cloud_machine" "k8s-master" {
  count         = var.master_count
  customer_id   = var.customer_id
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  image_id      = data.try-whitesky-cloud_image.image.image_id
  memory        = var.memory
  vcpus         = var.vcpus
  disk_size     = var.disksize
  name          = "master-${count.index}-${ try-whitesky-cloud_cloudspace.cs.location}"
  description   = "${var.vm_description} master"
  userdata      = "users: [{name: ansible, shell: /bin/bash, ssh-authorized-keys: [${var.ssh_key_pub}]}, {name: root, shell: /bin/bash, ssh-authorized-keys: [${var.ssh_key_pub}]}]"
}

resource "null_resource" "provision-local" {
  triggers = {
    build_number = timestamp()
  }
  # this is added to ensure connectivety on the management node
  provisioner "remote-exec" {
    inline = ["echo"]
  }
  provisioner "local-exec" {
    command = "ssh-keygen -R [${ try-whitesky-cloud_cloudspace.cs.external_network_ip}]:${ try-whitesky-cloud_port_forwarding.mgt-ssh.public_port} || true"
  }
  provisioner "local-exec" { 
    command = "ssh-keyscan -H -p ${ try-whitesky-cloud_port_forwarding.mgt-ssh.public_port} ${ try-whitesky-cloud_cloudspace.cs.external_network_ip} >> ${var.known_hosts_location}"
  }
  connection {
    type        = "ssh"
    user        = "ansible"
    host        = try-whitesky-cloud_cloudspace.cs.external_network_ip
    port        = try-whitesky-cloud_port_forwarding.mgt-ssh.public_port
    private_key = file(var.ssh_key_private_location)
  }
}

# configure user access on master nodes
resource "null_resource" "provision-k8s-master" {
  count = var.master_count
  triggers = {
    build_number = timestamp()
  }
  provisioner "file" {
    content     = "ansible    ALL=(ALL:ALL) NOPASSWD: ALL"
    destination = "/etc/sudoers.d/90-ansible"
  }
  connection {
    type         = "ssh"
    user         = "root"
    host         = try-whitesky-cloud_machine.k8s-master.*.interfaces[count.index][0]["ip_address"]
    private_key  = file(var.ssh_key_private_location)
    bastion_user = "ansible"
    bastion_host = try-whitesky-cloud_cloudspace.cs.external_network_ip
    bastion_port = try-whitesky-cloud_port_forwarding.mgt-ssh.public_port
  }
}

## Worker machines
resource "try-whitesky-cloud_machine" "k8s-worker" {
  count         = var.worker_count
  customer_id   = var.customer_id
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  image_id      = data.try-whitesky-cloud_image.image.image_id
  memory        = var.memory
  vcpus         = var.vcpus
  disk_size     = var.disksize
  name          = "worker-${count.index}-${ try-whitesky-cloud_cloudspace.cs.location}"
  description   = "${var.vm_description} node"
  userdata      = "users: [{name: ansible, shell: /bin/bash, ssh-authorized-keys: [${var.ssh_key_pub}]}, {name: root, shell: /bin/bash, ssh-authorized-keys: [${var.ssh_key_pub}]}]"
}
resource "try-whitesky-cloud_disk" "worker-disk" {
  count       = var.worker_count
  vm_id       = try-whitesky-cloud_machine.k8s-worker.*.id[count.index]
  location    = var.location
  customer_id = var.customer_id
  disk_name   = "data-worker-${count.index}-${ try-whitesky-cloud_cloudspace.cs.location}"
  description = "Disk created by terraform"
  size        = 10
  type        = "DATA"
  iops        = 2000
}

# Kubernetes Port forwards

resource "try-whitesky-cloud_port_forwarding" "k8s-master-api" {
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  customer_id   = var.customer_id
  vm_id         = try-whitesky-cloud_machine.k8s-master.*.id[0]
  public_port   = 6443
  local_port    = 6443
  protocol      = "tcp"
}
resource "try-whitesky-cloud_port_forwarding" "k8s-worker-0-http" {
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  customer_id   = var.customer_id
  vm_id         = try-whitesky-cloud_machine.k8s-worker.*.id[0]
  public_port   = 80
  local_port    = 30000
  protocol      = "tcp"
}
resource "try-whitesky-cloud_port_forwarding" "k8s-worker-0-https" {
  cloudspace_id = try-whitesky-cloud_cloudspace.cs.id
  customer_id   = var.customer_id
  vm_id         = try-whitesky-cloud_machine.k8s-worker.*.id[0]
  public_port   = 443
  local_port    = 30001
  protocol      = "tcp"
}

resource "null_resource" "provision-k8s-worker" {
  count         = var.worker_count
  triggers = {
      build_number = timestamp()
  }
  # configure access for ansible user
  provisioner "file" {
    content      = "ansible    ALL=(ALL:ALL) NOPASSWD: ALL"
    destination = "/etc/sudoers.d/90-ansible"
  }
  depends_on = [try-whitesky-cloud_disk.worker-disk]
  # Download script to move the data of the /var directory
  # to the partion /dev/vdb. The system is rebooted
  provisioner "remote-exec" {
    inline = [
      "mkdir -p /home/ansible/scripts",
      "cd /home/ansible/scripts && wget https://raw.githubusercontent.com/gig-tech/tutorial-help-scripts/master/kubernetes-cluster-deployment/move-var.sh",
      "sudo -S bash /home/ansible/scripts/move-var.sh",
    ]
  }
  connection {
    type     = "ssh"
    user         = "root"
    host     = try-whitesky-cloud_machine.k8s-worker.*.interfaces[count.index][0]["ip_address"]
    private_key  = file(var.ssh_key_private_location)
    bastion_user = "ansible"
    bastion_host = try-whitesky-cloud_cloudspace.cs.external_network_ip
    bastion_port = try-whitesky-cloud_port_forwarding.mgt-ssh.public_port
  }
}

# Ansible Inventory
resource "local_file" "AnsibleInventory" {
 content = templatefile("inventory.tmpl",
 {
  masters = try-whitesky-cloud_machine.k8s-master.*,
  workers = try-whitesky-cloud_machine.k8s-worker.*
 }
 )
 filename = "inventory"
}

# Copy inventory to mgmt-vm
resource "null_resource" "prepare-ansible" {
  triggers = {
    build_number = timestamp()
  }
  depends_on = [local_file.AnsibleInventory]
  provisioner "remote-exec" {
    inline = ["apt-get update", 
              "apt-get install python3-pip git -y"]
  }
  provisioner "file" {
    source      = "ansible"
    destination = "/root"
  }
  provisioner "file" {
    source      = "helm"
    destination = "/root"
  }
  provisioner "file" {
    source      = "inventory"
    destination = "/root/ansible/inventory.ini"
  }
  connection {
    type        = "ssh"
    user        = "root"
    host        = try-whitesky-cloud_cloudspace.cs.external_network_ip
    port        = try-whitesky-cloud_port_forwarding.mgt-ssh.public_port
    private_key = file(var.ssh_key_private_location)
  }
}